#!/bin/bash 

# tkl2lxc script by Adrian E. Moya G. 2011

usage() {
cat<<EOF
Syntax: $(basename $0) imagename.iso 
Creates a LXC container filesystem from imagename.iso
EOF
    exit 1
}

if [ "$#" != "1" ]; then
    usage
fi

extract_fs()
{
    isoimage=$1
    imagename=`echo $isoimage | sed -n 's|\(.*\)\.iso|\1|p'`

    tklpatch-extract-iso $isoimage
    rm -Rf $imagename.cdroot/
    mv $imagename.rootfs/ rootfs/
}

configure_ubuntu()
{
    rootfs=$1

    # suppress log level output for udev
    sed -i "s/=\"err\"/=0/" $rootfs/etc/udev/udev.conf

    # remove jobs for consoles 5 and 6 since we only create 4 consoles in
    # this template
    rm -f $rootfs/etc/init/tty{5,6}.conf

    return 0
}


trim()
{
    rootfs=$1

    # provide the lxc service
    cat <<EOF > $rootfs/etc/init/lxc.conf
# fake some events needed for correct startup other services

description     "Container Upstart"

start on startup

script
        rm -rf /var/run/*.pid
        rm -rf /var/run/network/*
        /sbin/initctl emit stopped JOB=udevtrigger --no-wait
        /sbin/initctl emit started JOB=udev --no-wait
end script
EOF

    # fix buggus runlevel with sshd
    cat <<EOF > $rootfs/etc/init/ssh.conf
# ssh - OpenBSD Secure Shell server
#
# The OpenSSH server provides secure shell access to the system.

description	"OpenSSH server"

start on filesystem
stop on runlevel [!2345]

expect fork
respawn
respawn limit 10 5
umask 022
# replaces SSHD_OOM_ADJUST in /etc/default/ssh
oom never

pre-start script
    test -x /usr/sbin/sshd || { stop; exit 0; }
    test -e /etc/ssh/sshd_not_to_be_run && { stop; exit 0; }
    test -c /dev/null || { stop; exit 0; }

    mkdir -p -m0755 /var/run/sshd
end script

# if you used to set SSHD_OPTS in /etc/default/ssh, you can change the
# 'exec' line here instead
exec /usr/sbin/sshd
EOF

    cat <<EOF > $rootfs/etc/init/console.conf
# console - getty
#
# This service maintains a console on tty1 from the point the system is
# started until it is shut down again.

start on stopped rc RUNLEVEL=[2345]
stop on runlevel [!2345]

respawn
exec /sbin/getty -8 38400 /dev/console
EOF

    cat <<EOF > $rootfs/lib/init/fstab
# /lib/init/fstab: cleared out for bare-bones lxc
EOF

    # reconfigure some services
    chroot $rootfs locale-gen en_US.UTF-8
    chroot $rootfs update-locale LANG=en_US.UTF-8

    # remove pointless services in a container
    chroot $rootfs /usr/sbin/update-rc.d -f ondemand remove

    chroot $rootfs /bin/bash -c 'cd /etc/init; for f in $(ls u*.conf); do mv $f $f.orig; done'
    chroot $rootfs /bin/bash -c 'cd /etc/init; for f in $(ls tty[2-9].conf); do mv $f $f.orig; done'
    chroot $rootfs /bin/bash -c 'cd /etc/init; for f in $(ls plymouth*.conf); do mv $f $f.orig; done'
    chroot $rootfs /bin/bash -c 'cd /etc/init; for f in $(ls hwclock*.conf); do mv $f $f.orig; done'
    chroot $rootfs /bin/bash -c 'cd /etc/init; for f in $(ls module*.conf); do mv $f $f.orig; done'

}

tkl_extras()
{
    rootfs=$1

    # Preseed default values
    echo "ROOT_PASS=turnkey" >> $ROOTFS/etc/default/inithooks
    echo "HUB_APIKEY=SKIP" >> $ROOTFS/etc/default/inithooks
    echo "SEC_UPDATES=SKIP" >> $ROOTFS/etc/default/inithooks

    # VT Not working at init script for inithooks
    sed -i "s/fgconsole/#fgconsole/" rootfs/etc/init.d/inithooks

}

if [ "$(id -u)" != "0" ]; then
    echo "This script should be run as 'root'"
    exit 1
fi

isoimage=$1
release=lucid
arch=i386

if ! [ -f $isoimage ]; then
    echo "$isoimage doesn't exists"
    exit 1
fi

if [ -d rootfs/ ]; then
    echo "rootfs/ already exists. Remove it to run the script again"
    exit 1
fi

extract_fs $isoimage

configure_ubuntu rootfs 
if [ $? -ne 0 ]; then
    echo "failed to configure turnkeylinux for a container"
    exit 1
fi

trim rootfs
if [ $? -ne 0 ]; then
    echo "failed to trim turnkeylinux container"
    exit 1
fi

tkl_extras rootfs
if [ $? -ne 0 ]; then
    echo "failed to apply tkl_extras to container"
    exit 1
fi

echo "Finished creating lxc rootfs for turnkeylinux"
